""" A Python class to operate the FlexControl device from FlexRadio.
    It contains code to:
    - Listen to the FlexControl and capture control event codes sent to the
    host over USB.
    - Control the lights on the FlexControl.
    - Terminate the program, with a double-click of the control knob.
    
    The code currently does nothing with the event codes except to print them
    as received from the device. To be useful, the codes need to be sent
    somewhere for processing.
    
    NOTES:
    - This program is written for use on a Linux host, which is reflected in
    the default serial device. To use with another system, change the connection
    port as needed.
    - It should be treated as sample code. It needs code for error-handling
    when accessing the USB device, and possibly in other places, before it can
    be considered ready for production use,
    - This program was made possible by information obtained from the blog of
    AA6E at: https://blog.aa6e.net/2015/01/flexcontrol-for-linux-hamr.html.
    Thank you Martin!
    
    Version: 1.0.0
    Copyright 2020 by Russell Schoof, KC7MM. All rights reserved.
    This program is distributed under the terms of the GNU General Public
    License, verion 3, as specified at https://www.gnu.org/licenses/gpl-3.0.html
"""

import serial, time

class FlexControl():
    def __init__(self, in_port='/dev/ttyACM0'):
        """ Prepare the computer to work with the device.
            PARAMETER in_port designates the serial device the FlexControl is
            plugged into. Defaults to dev/ttyACM0 on Linux.
        """
        # Open the port 
        self.flexctl = serial.Serial(in_port)    
        self.configLights()      # Configure light control
        self.flashAllLights(2)      # Flash the lights to show startup
        self.setOneLight(self.ONOFFLIGHT, self.LIGHTON) # Set the On light

    def runControl(self):
        """ Puts the flexcontrol into operating mode with an event loop.
            The function receives codes generated by operator manipulation of
            the knob and buttons on the device.

            NOTE: This function just prints the code; it meeds to be modified
            to send it someplace where it can be processed. 
        """
        # Start event loop to read input codes from the control
        while True:
            in_code = self.getEventCode()
            if in_code == 'C':          # Check for code to shut down
                self.flashAllLights(3)  # Flash the lights,
                break                   #  and exit event loop to shut down.
            else:
                # Process the code
                print(in_code)  #### DEBUG: TEMPORARY TEST CODE
                # TODO: send the code for processing

    def getEventCode(self):
        """ Gets the code for a single event on the FlexControl.
            RETURN: A string consisting of the code.
        """
        in_chars = []   # Array to accumulate code characters
        # Read from the port until end-of-code sysmbol is received.
        while True:
            f = self.flexctl.read().decode()
            if not f == ';':    # Look for end of code symbol: a semicolon
                in_chars.append(f)
            else:
                break
        # Return the code characters, less end-of-code symbol
        return(''.join(in_chars))

    ### Begin light control functions. ##################
    def setOneLight(self, light_to_set, on_or_off):
        """ Turns one light on or off without affecting others.
            PARAMETER light_to_set is the light that is to be turned on or off.
                The expected value is one of the lights defined in the function
                self.configLights.
            PARAMETER on_or_off is the desired setting. The expected value is 
                either self.LIGHTON or self.LIGHTOFF.
        """
        # Set the light's value in the light state array
        self.LIGHT_STATE[light_to_set] = on_or_off
        # Send the current light states to the device
        self.flexctl.write(self.LIGHT_STATE)

    def flashAllLights(self, times=1):
        """ Flash all lights a designated number of times. 
            PARAMTER times is the number of times to flash.
        """
        # Turn all lights off for proper flash effect
        self.flexctl.write(self.ALLLIGHTSOFF)
        # Now flash them the designated number of times
        for i in list(range(times)):
            self.flexctl.write(self.ALLLIGHTSON)
            time.sleep(.25)
            self.flexctl.write(self.ALLLIGHTSOFF)
            time.sleep(.25)

    def configLights(self):
        """ Configure the program to control lights on the device.
        """
        # Initialize light status variable to default value of all lights off.
        self.LIGHT_STATE = bytearray('I000;'.encode())
        # Set light on/off constant values
        self.LIGHTON = ord('1')
        self.LIGHTOFF = ord('0')
        self.ALLLIGHTSON = 'I111;'.encode()
        self.ALLLIGHTSOFF = 'I000;'.encode()
        # Set device lights' position in the light status variable
        self.MODELIGHT = 1      # Ordinal in LIGHT_STATE array
        self.ONOFFLIGHT = 2     # Ordinal in LIGHT_STATE array
        self.TOGGLELIGHT = 3    # Ordinal in LIGHT_STATE array

    ### End light control functions. ##################

if __name__ == '__main__':
    flexcontrol = FlexControl()
    flexcontrol.runControl()
